//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQml 2.12
import QtQuick 2.12
import QtQuick.Layouts 1.12
import QtQuick.Dialogs 1.3

import wesual.Ui 1.0
import wesual.Util 1.0

import wesual.Ui.Private 1.0

import "private"

UiColorPickerBase {
    id : uiColorPicker

    property var palette : []

    function addColorToPalette() {
        if (uiColorPicker.writablePalettes.length === 1) {
            uiColorPicker.writablePalettes[0].insertColor(
                        uiColorPicker.color);
        } else if (uiColorPicker.writablePalettes.length > 1) {
            paletteChoiceDialog.color = uiColorPicker.color;
            paletteChoiceDialog.show();
        }
    }

    implicitWidth : 352
    implicitHeight : content.implicitHeight

    layer.effect  : ShaderEffect {
        fragmentShader : "
            uniform lowp sampler2D source;
            uniform lowp float qt_Opacity;

            varying highp vec2 qt_TexCoord0;

            void main() {
                lowp vec4 p = texture2D(source, qt_TexCoord0);
                lowp float g = dot(p.xyz, vec3(0.344, 0.5, 0.156));
                gl_FragColor = vec4(g, g, g, p.a) * .5 * qt_Opacity;
            }"
    }

    mode : modeSetting.value

    states : State {
        name : "disabled"
        when : !uiColorPicker.enabled

        PropertyChanges {
            target : uiColorPicker

            layer.enabled : true
        }
    }

    Setting {
        id : modeSetting

        domain : "Volatile.ColorPicker"
        key : "mode"
        defaultValue : 1
    }

    Component {
        id : colorEditor

        UiColorEditor {
            color : uiColorPicker.color

            indeterminate : uiColorPicker.indeterminate
            onUpdateRequested : uiColorPicker.updateColor(newColor)

            additionalControls : UiToolButton {
                iconName : "add"
                visible : uiColorPicker.writablePalettes.length > 0
                onClicked : addColorToPalette()
            }

            UiContextMenu {
                enabled : uiColorPicker.writablePalettes.length > 0 &&
                          !uiColorPicker.indeterminate

                UiMenuItem {
                    text : qsTrId("26ee2e66a77d9815")
                    onTriggered : addColorToPalette()
                }
            }
        }
    }

    Component {
        id : palettePanel

        Column {
            id : palettes

            spacing : 5
            padding : 3

            // Legacy transparent patch - will be removed eventually.
            // Only needed in one template location ...
            Image {
                id : transparentPatch

                width : 22
                height : 22
                source : "qrc:/ui/images/color-transparent.png"
                visible : uiColorPicker.usesTransparent

                states : [
                    State {
                        name : "selected"
                        when : uiColorPicker.transparent

                        PropertyChanges {
                            target : inner
                            opacity : 1
                        }
                        PropertyChanges {
                            target : outer
                            opacity : 1
                            color : UiColors.getColor(UiColors.SelectionGreen)
                        }
                    },
                    State {
                        name : "hovered"
                        when : mouseArea.containsMouse &&
                               !uiColorPicker.transparent
                        PropertyChanges {
                            target : inner
                            opacity : 1
                        }
                        PropertyChanges {
                            target : outer
                            opacity : 1
                        }
                    }
                ]

                MouseArea {
                    id : mouseArea

                    anchors.fill : parent

                    hoverEnabled : true
                    onClicked : uiColorPicker.updateColor("transparent")
                }

                UiBorder {
                    id : inner

                    anchors.fill : parent

                    style : UiBorder.Solid
                    opacity : 0

                    borderWidth : 2
                    color : UiColors.getColor(UiColors.White)
                }

                UiBorder {
                    id : outer

                    anchors.fill : parent
                    anchors.margins : -3

                    style : UiBorder.Solid
                    opacity : 0

                    borderWidth : 3
                    color : UiColors.getColor(UiColors.HoverGreen)
                }
            }

            UiContextMenu {
                enabled : uiColorPicker.writablePalettes.length > 0 &&
                          !uiColorPicker.indeterminate

                UiMenuItem {
                    text : qsTrId("ca86ff8e8635182e")
                    onTriggered : {
                        if (uiColorPicker.writablePalettes.length === 1) {
                            colorDialog.open(function(color) {
                                uiColorPicker.writablePalettes[0].insertColor(
                                            color);
                            });
                        } else {
                            colorDialog.label = text;
                            paletteChoiceDialog.color = null;
                            paletteChoiceDialog.show();
                        }
                    }
                }
            }

            Repeater {
                model : uiColorPicker ? uiColorPicker.palettes : []

                Component.onDestruction : {
                    model = [];
                }

                delegate : UiPaletteItem {
                    id : paletteItem

                    palette : modelData
                    colorPicker : uiColorPicker
                    anchors {
                        left : palettes.left
                        right : palettes.right
                        leftMargin : 3
                    }

                    UiContextMenu {
                        id : menu

                        property Item colorPatch

                        onContextMenuRequested : {
                            var item = paletteItem.childAt(
                                        position.x, position.y);
                            if (item && "colorIndex" in item) {
                                colorPatch = item;
                            } else {
                                colorPatch = null;
                            }
                        }

                        enabled : !palette.readOnly &&
                                  !uiColorPicker.indeterminate

                        UiMenuItem {
                            text : qsTrId("211f62ad4dd54d5b")
                            visible : menu.colorPatch !== null
                            onTriggered : {
                                colorDialog.label = text;
                                colorDialog.color = menu.colorPatch.color;
                                colorDialog.open(function(color) {
                                    paletteItem.palette.setColor(
                                                color,
                                                menu.colorPatch.colorIndex,
                                                menu.colorPatch.swatchIndex)
                                });
                            }
                        }

                        UiMenuItem {
                            text : qsTrId("2c902836a1b870d9")
                            visible : menu.colorPatch !== null
                            onTriggered : {
                                paletteItem.palette.removeColor(
                                            menu.colorPatch.colorIndex,
                                            menu.colorPatch.swatchIndex)
                            }
                        }

                        UiMenuItem {
                            text : qsTrId("ca86ff8e8635182e")
                            onTriggered : {
                                colorDialog.label = text;
                                colorDialog.open(function(color) {
                                    paletteItem.palette.insertColor(color);
                                });
                            }
                        }
                    }
                }
            }
        }
    }

    RowLayout {
        id : content

        anchors.fill : parent
        spacing : 2

        Column {
            Layout.alignment : Qt.AlignTop

            spacing : 4

            UiButton {
                width : 30
                height : 30
                enabled : !uiColorPicker.transparent
                onClicked : modeSetting.update((modeSetting.value + 1) % 2)

                UiIcon {
                    iconName : [ "color-swatches", "color-mixer" ][mode]
                    disabled : uiColorPicker.transparent ? 1 : 0
                    anchors.centerIn : parent
                }
            }

            UiToolButton {
                iconName : "eyedropper"
                onClicked : uiColorPicker.pickColor()
            }

            UiBorder {
                width : 30
                height : 30

                borderWidth : 3
                color : UiColors.getColor(UiColors.SelectionGreen)
                style : UiBorder.Solid
                visible : !uiColorPicker.transparent &&
                          uiColorPicker.hasCustomColor &&
                          uiColorPicker.mode === UiColorPickerBase.Swatches

                Rectangle {
                    width : 20
                    height : 20
                    color : uiColorPicker.color
                    anchors.centerIn : parent
                }

                UiContextMenu {
                    enabled : uiColorPicker.writablePalettes.length > 0 &&
                              !uiColorPicker.indeterminate

                    UiMenuItem {
                        text : qsTrId("26ee2e66a77d9815")
                        onTriggered : addColorToPalette()
                    }
                }
            }
        }

        Loader {
            Layout.fillHeight : true
            Layout.fillWidth : true

            sourceComponent : [ colorEditor, palettePanel ][mode]
        }
    }

    UiColorDialog {
        id : colorDialog

        property var cb

        function open(cb) {
            colorDialog.cb = cb;
            show();
        }

        onAccepted : {
            if (cb) {
                cb(result);
            }
        }
    }

    Component {
        id : paletteChoice

        UiChoice {
            property var palette

            text : palette.name
            icon : palette.icon
            onClicked : {
                if (!paletteChoiceDialog.color) {
                    colorDialog.open(function(color) {
                        palette.insertColor(color);
                    });
                } else {
                    palette.insertColor(paletteChoiceDialog.color);
                }
            }
        }
    }

    UiChoiceDialog {
        id : paletteChoiceDialog

        property var color : null

        label : qsTrId("ca86ff8e8635182e")

        choices : {
            var choices = [];

            for (var i = 0; i < uiColorPicker.writablePalettes.length; ++i) {
                var palette = uiColorPicker.writablePalettes[i];
                choices.push(paletteChoice.createObject(paletteChoiceDialog, {
                    "palette" : palette
                }));
            }

            return choices;
        }

    }
}
